function values = zz_colalign(values,formats,adjustment)

% values = zz_colalign(values,formats,adjustment)
% ver 221102 SAM
% 
% function for converting cell arrays to strings for formatted output
%
% "values" is a cell array of values (real numbers, strings, logicals). It
% can be a mixture of types.
%
% "formats" is either a cell array of formats used in num2str or a single
% value. In case of a single value:
% - a numeric input: returns formatted strings with significant digits
% equal to the numeric input. In cases where a number cannot be displayed
% without using exponential notation, all numeric values are displayed with 
% exponential notation.
% - a format string, e.g. '%.4f'.
% - missing or empty, applies '%.5g' with the same behaviour regarding 
% exponential notation as single numeric input. 
% 
% 'adjustment' is 'decimal' (default), 'right', 'left' or 'center'
% 
% the returned cell array is always of dimension (n x 1). 

% ensure values are column
values = values(:);

% set default alignment 
if nargin<3
    adjustment = 'decimal';
end
if isempty(adjustment)
    adjustment = 'decimal';
end

% convert empties to char
empties = cellfun(@isempty,values);
values(empties) = {''};

nums = cellfun(@isnumeric,values);
logicals = cellfun(@islogical,values);
chars = cellfun(@ischar,values) & ~empties;

% set the standard format
stdformat = '%#.5g'; % default

if nargin<2
    formats = [];
end
if isempty(formats)
    formats = 5;
end
if isnumeric(formats)
    % finhd max of finite values, ignore NaN and Inf
    finitevals = abs(cell2mat(values(nums)));
    finitevals(~isfinite(finitevals)) = 0;
    maxval = max(finitevals);
    % find max number of digits
    maxdig = max(floor(log10(maxval))+1,1);
    if maxdig>formats
        stdformat = ['%#.' num2str(formats) 'e'];
    else
        stdformat = ['%#.' num2str(formats) 'g'];
    end
end
if ischar(formats) % single format specification e.g. '%.4f'
    stdformat = formats;
end

% delete format input unless cell
if ~iscell(formats)
    formats = [];
end

% set counters
nchar = zeros(numel(values),1);
intlength = nchar;
declength = nchar;

% convert
for x = 1:numel(values)

    if nums(x)
        if numel(formats) >= x && ~isempty(formats(x))
            format = formats{x};
        else
            format = stdformat;
        end

        if isfinite(values{x})

            values{x} = num2str(values{x},format);
            %values{x} = sprintf(format,values{x}); % sprintf does no strtrim if fieldwidth is specified
            nchar(x) = numel(values{x});
            intlength(x) = nchar(x);

            k = strfind(values{x},'.');
            if ~isempty(k)
                intlength(x) = k-1;
                declength(x) = nchar(x)-k+1;
            end

        else

            values{x} = num2str(values{x},format);
            nchar(x) = numel(values{x});

        end

    end

    if logicals(x)
        if values{x}
            values{x} = 'true';
        else
            values{x} = 'false';
        end
    end

    nchar(x) = numel(values{x});
    
end

switch adjustment

    case {'decimal','dec'}

        % calculate widths
        intwidth = max(intlength);
        numwidth = intwidth + max(declength);
        totwidth = max(max(nchar),numwidth);
        dstr = repmat(' ',1,totwidth);

        % position strings within dstr
        for x = 1:numel(values)

            ostr = dstr;

            if nums(x)

                if intlength(x) + declength(x) > 0
                    posv = (1:nchar(x)) + (intwidth-intlength(x));
                else
                    offs = max(numwidth,nchar(x));
                    posv = (offs-nchar(x)+1):offs;
                end
                ostr(posv) = values{x};
                values{x} = ostr;
                continue
            end

            if chars(x)
                ostr(1:nchar(x)) = values{x};
                values{x} = ostr;
                continue                
            end

            if logicals(x)
                offs = max(numwidth,5);
                posv = (offs-nchar(x)+1):offs;
                ostr(posv) = values{x};
                values{x} = ostr;             
            end

        end

    case 'right'

        intwidth = max(intlength);
        numwidth = intwidth + max(declength);
        totwidth = max(max(nchar),numwidth);
        dstr = repmat(' ',1,totwidth);

        % position strings within dstr
        for x = 1:numel(values)

            ostr = dstr;

            if nums(x) 
                posv = (numwidth-nchar(x)+1):numwidth;
                ostr(posv) = values{x};
                values{x} = ostr;
                continue
            end

            if chars(x)
                ostr(1:nchar(x)) = values{x};
                values{x} = ostr;
                continue
            end

            if logicals(x)
                offs = max(numwidth,5);
                posv = (offs-nchar(x)+1):offs;
                ostr(posv) = values{x};
                values{x} = ostr;
            end

        end

    case 'left'

        totwidth = max(nchar);
        dstr = repmat(' ',1,totwidth);

        % position strings within dstr
        for x = 1:numel(values)

            ostr = dstr;
            ostr(1:nchar(x)) = values{x};
            values{x} = ostr;

        end   

    case 'center'

        totwidth = max(nchar);
        medwidth = totwidth/2;
        dstr = repmat(' ',1,totwidth);

        % position strings within dstr
        for x = 1:numel(values)

            ostr = dstr;
            posv = (1:nchar(x)) + floor(medwidth -nchar(x)/2);
            ostr(posv) = values{x};
            values{x} = ostr;

        end

end % switch