function dataout=zz_tabletool(call,arg1,arg2,arg3)
% function dataout=zz_tabletool(call,arg1,arg2,arg3)
% ver 200308 SAM
%
% ======= CHROMBOX TABLETOOL =======
% Tool for handling data in tabulated form
%
% CREATION:
% A table is created by the following command:
% xx_tabletool('Create',T)
% T is a struct containing the following required fields:
% - TableName: String used as identifier for the table
% - Position: Four element vector of positiondata in normalized units
% - Data: Cell array of data. can contain strings, numbers, binary values or RGB color vectors
%
% The following optional fields can be specified:
%
% - ViewPref: Default view, numeric 1-3
%   - 1: Will use Matlab uitable if supported by the Matlab version
%   - 2: Will create table without the uitable function
%   - 3: List view
%
% - Allowed views: Views that are allowed, vector with numbers 1-3
%
% - ColumnFormat: As specified for Matlab uitable. In addition the
% following column formats can be used:
%   - string: same as char
%   - binary: same as logical
%   - excl_logical or excl_binary: as binary but only one row can be
%   selected
%   - text: static text
%   - color: RGB color vector. Only supported by view 2.
%   - an integer specifying the precision of numeric values
%   - a formt string specifying the format of numeric values (see fprintf)
% The default column format is 'char'
% - def_ColumnFormat: chnges the default column format
% - Pusbutton, input data is text on buttons, use getlastcell to determine
% which button that was pressed.
%
% ListSelCol: a single integer referring to a column with logical or
% excl_logical values. Applied for selecting rows in list view.
% ShowListSel: Show/hide 'x' marking selection in list
% Default is 1 (show). Faster performance if selection is hidden.
%
% - ColumnWidth: cell array containing column widths in pixels, a single
% number (will be applied to all) or 'auto': Default is 'auto' for view 1
% and 100 px for view 2.
% - def_ColumnWidth: changes default column width
%
% - RowName: 'numeric' cell array of row names or [] (no row names), default is numeric
% - ColumnName: 'numeric' or cell array of column names, default is numeric
%
% - ColumnEditable: A single logical value (for all columns) or a cell
% array of locical values for each column specifying if columns can be
% edited or not.
%
% - CellEditCallback: Callback that is run when a cell is edited
% 
% - FontSize: Font size in units
% 
% - MaxVal: 1 by number of columns cell array specifying max allowable value that can be set in
% each column. Default is Inf.
% - MinVal: 1 by number of columns cell array specifying min allowable value that can be set in
% each column. Default is -Inf.
% - MaxValCell and MinValCell: Number of rows by number of columns cell array specifying max/min 
% values of eac cell. Will ovveride MaxVal and MinVal.
% - AllowNan/AllowNanCell: cell array of binary values specifying if NaNs
% are accepted. Dimensions as for MaxVal/MaxValCell. Default is 0 (NaNs
% forbidden)
% - AllowEmpty/AllowEmptyCell: Same as AllowNan/AllowNanCell but specifies if
% empty values are allowed. 
%
% - HeaderText: Table header. Cell array of strings
% - FooterText: Table footer. Cell array of strings
%
% - ReportAddress: Path/filename (excluding extension) for reports
%
% - T.EmptyMessage: Message to display if empty table. String or cell of
% strings. 'No Data' is displayed if unset. Set to '' or [] to display
% nothing
%
% The following parameters can be set for view 2:
% - RowNamesWidth: Width of row names in pixels
% - ColumnNamesHeight: Height of column names
% - HorizontalExpansion: 1 or 0. 1 will expand visible columns to fit
% available space
% 
% - Popup: Creates table in new window, position is then window position in screen
% The following parameters can be set for popuptables
% - PopupTitle: Title of figure (informative only)
% - PopupCloseRequestFcn: function to be called when figure is closed 
% - PopupResizable: 1=on, 0=off
% - PopupModal: 1=modal window
%

%disp(['--- zz_tabletool_190813 ' num2str(call)]); 

if nargin<4 arg3=[]; end
if nargin<3 arg2=[]; end
if nargin<2 arg1=[]; end
dataout=[];T=[];

% CALLS:
% 1 - Creates table
% 2 - Returns associated table struct
% 3 - returns field in associated table struct
% 4 - returns value from last edit (col, row, value)
% 5 - returns data from last edit
% 90 -clear all tables

% Settings on T
% T.def_ColumnFormat             % Default column format
% T.ColumnFormat                 % cell of column formats
% Valid column fomats are: 'char', integer (denotes precision), formatstr,
% cell array of strings (popupmenu), 'logical'/'binary', 'int' (integer)

% Convert string input til numeric (internal)
if ischar(call) && strcmpi(call,'create') call=1; end %#ok<*SEPEX>
if ischar(call) && strcmpi(call,'exist') call=0.5; end
if ischar(call) && strcmpi(call,'delete') call=0; end
if ischar(call) && strcmpi(call,'setcell') call=1.1; end
if ischar(call) && strcmpi(call,'setrow') call=1.2; end
if ischar(call) && strcmpi(call,'setcol') call=1.3; end
if ischar(call) && strcmpi(call,'setdata') call=1.4; end
if ischar(call) && strcmpi(call,'setparam') call=1.6; end
if ischar(call) && strcmpi(call,'getlastcell') call=2.1; end
if ischar(call) && strcmpi(call,'getcell') call=2.15; end
if ischar(call) && strcmpi(call,'getlastrow') call=2.2; end
if ischar(call) && strcmpi(call,'getrow') call=2.25; end
if ischar(call) && strcmpi(call,'getlastcol') call=2.3; end
if ischar(call) && strcmpi(call,'getcol') call=2.35; end
if ischar(call) && strcmpi(call,'getdata') call=2.4; end
if ischar(call) && strcmpi(call,'getall') call=2.5; end 
if ischar(call) && strcmpi(call,'getparam') call=2.6; end 

% ========= INPUT DATA ========

if ischar(call) disp('ERROR: zz_tabletool: parameter not recognized'); call=-99; end

if call==0
    %disp('---- delete table ---')
    popwin=findobj('Tag',['PopupTable_' arg1]);
    if isempty(popwin)
        cleartable(arg1)
    else
        % popups, does not run closerequestfcn, 
        delete(findobj('Tag',['PopupTable_' arg1])) 
    end
end

% Check if table exists
if call==0.5
    tableh=findobj('Tag',['TT_main_' arg1]);
    if isempty(tableh) 
        dataout=0;
    else
        dataout=1;
    end
end
    
% Create table
if call==1
    if ~isfield(arg1,'Data') || isempty(arg1.Data)
        T=initempty(arg1); % Display empty field and use for debugging
        return
    end    
    T=initialize(arg1); tabletype=1; 
    if isfield(T,'ViewPref') tabletype=T.ViewPref; end
    switch tabletype
        case 1  
            try 
                T=initnew(T); 
            catch
                T=initold(T); 
            end
        case 2
            T=initold(T);
        case 3
            T=initlist(T);
    end
end

% Send data to table
if call>1 && call<2
    tablename=arg1;
    tableh=findobj('Tag',['TT_main_' tablename]);
    T=get(tableh,'UserData');
    if isempty(T) disp('TABLETOOL ERROR: TABLE NOT FOUND'); call=-99; end
end

if call==1.1 && ~isempty(tableh)
    %disp('---- update cell ----')  
    T.Data{arg2.rowindex,arg2.colindex}=arg2.val;
    set(tableh,'UserData',T)
    updatevals(tablename)
end

if call==1.2 && ~isempty(tableh)
    %disp('---- update row ----')
    [T.Data{arg2.rowindex,:}]=deal(arg2.val{:});
    set(tableh,'UserData',T)
    updatevals(tablename)
end
if call==1.3 && ~isempty(tableh)
    %disp('---- update col ----')  
    [T.Data{:,arg2.colindex}]=deal(arg2.val{:});
    set(tableh,'UserData',T)
    updatevals(tablename)
end
if call==1.4 && ~isempty(tableh)
    %disp('---- update data ----')
    T.Data=arg2.val;
    set(tableh,'UserData',T)
    updatevals(tablename)
end
if call==1.5 && ~isempty(tableh)
    %disp('---- update T not implemented ----')
end
if call==1.6 && ~isempty(tableh)
    %disp('---- Set parameter ----')  
    T=updateparameter(T,arg2,arg3);
end  
        
% ========= REQUEST DATA ========
if call>2 && call<3
    tablename=arg1;
    tableh=findobj('Tag',['TT_main_' tablename]);
    T=get(tableh,'UserData');
end
                
if call==2.1
    %disp('---- request lastcell ----') 
    if length(T.lastedit_rowindex)>1
        for x=1:length(T.lastedit_rowindex)
            dataout(x).rowindex=T.lastedit_rowindex(x);  %#ok<*AGROW>
            dataout(x).colindex=T.lastedit_colindex(x); 
            dataout(x).datatype=T.lastedit_datatype;
            dataout(x).val=T.lastedit_value(x);
            dataout(x).previousval=T.lastedit_previousvalue(x);
            dataout(x).tablename=T.TableName;
        end
    else
        dataout.rowindex=T.lastedit_rowindex;
        dataout.colindex=T.lastedit_colindex;
        dataout.datatype=T.lastedit_datatype;
        dataout.val=T.lastedit_value;
        dataout.previousval=T.lastedit_previousvalue;
        dataout.tablename=T.TableName;        
    end
end
if call==2.15
    %disp('---- request cell ----')   
    dataout=T.Data{arg2(1),arg2(2)};
end

if call==2.2
    %disp('---- request lastrow ----') 
    if length(T.lastedit_rowindex)>1
        for x=1:length(T.lastedit_rowindex)
            dataout(x).rowindex=T.lastedit_rowindex(x);
            dataout(x).datatype=T.lastedit_datatype;
            dataout(x).val=T.Data(T.lastedit_rowindex(x),:);
            dataout(x).tablename=T.TableName;   
        end
    else
        dataout.rowindex=T.lastedit_rowindex;
        dataout.datatype=T.lastedit_datatype;
        dataout.val=T.Data(T.lastedit_rowindex,:);
        dataout.tablename=T.TableName;          
    end
end
if call==2.25
    %disp('--- request row ----')
    dataout=T.Data(arg2,:);
end

if call==2.3
    %disp('---- request lastcol ----')   
    dataout.colindex=T.lastedit_colindex(1);
    dataout.val=T.Data(:,T.lastedit_colindex(1));
    dataout.datatype=T.lastedit_datatype;
    dataout.tablename=T.TableName;   
end
if call==2.35
    %disp('--- request col ----')
    dataout=T.Data(:,arg2);
end

if call==2.4
    %disp('---- request Data ----')  
    dataout=T.Data;
end
if call==2.5
    %disp('---- request T ----')   
    dataout=T;
end
if call==2.6
    %disp('---- request parameter ----') 
    if isfield(T,arg2) dataout=T.(arg2); else dataout=[]; end
end

% ======== INTERNAL ========

% Sliders
if call==3.1
    userdat=get(gcbo,'userdata'); 
    T=get(userdat.tablehandle,'Userdata');    
    sv=1-get(gcbo,'Value'); % Slider val-1
    cv=round((sv*(T.O.maxfirstrowindex-1))+1); % Convert to index
    T.O.firstrowindex=cv;
    set(userdat.tablehandle,'Userdata',T);
    dispdata(T.TableName);
end
if call==3.2
    userdat=get(gcbo,'userdata'); 
    T=get(userdat.tablehandle,'Userdata');
    sv=get(gcbo,'Value'); % Slider val-1
    cv=round((sv*(T.O.maxfirstcolindex-1))+1); % Convert to index
    T.O.firstcolindex=cv;
    set(userdat.tablehandle,'Userdata',T);
    T=addobj(T.TableName); % add objects
    T=dispdata(T.TableName); % Display data
end

% Cellcallback
if call==4.1    
    %disp('--- cellcallback old---')
    newval=0; islegal=1; checknext=1; runupdate=0; userdat=get(gcbo,'UserData');
    tablename=userdat.tablename;
    bgrh=findobj('Tag',['TT_main_' tablename]);
    T=get(bgrh,'Userdata');
    
    rno=userdat.edit_rno+T.O.firstrowindex-1; % Refers to T.Data
    cno=userdat.edit_cno+T.O.firstcolindex-1; % Refers to T.Data
    oldval=T.Data{rno,cno};

    if checknext && strncmpi(T.datatype{cno},'numeric',3)
        checknext=0;
        newval=str2double(get(gcbo,'String'));           
        [islegal,newval]=checkval(T,newval,rno,cno);                        
    end
    if checknext && strncmpi(T.datatype{cno},'integer',3)
        checknext=0;
        newval=round(str2double(get(gcbo,'String'))); 
        [islegal,newval]=checkval(T,newval,rno,cno); 
    end    
    if checknext && strncmpi(T.datatype{cno},'char',3)
        newval=get(gcbo,'String');  
        checknext=0;
        [islegal,newval]=checkval(T,newval,rno,cno);
    end     
    if checknext && strncmpi(T.datatype{cno},'selection',3)
        checknext=0;
        str=get(gcbo,'String'); pos=get(gcbo,'Value'); newval=str{pos};
    end      
    if checknext && strncmpi(T.datatype{cno},'logical',3)
        checknext=0;
        newval=get(gcbo,'Value'); 
        if newval>0 newval=true; else newval=false; end
    end
    if checknext && strncmpi(T.datatype{cno},'excl_logical',3)
        checknext=0;
        previousset=cell2mat(T.Data(:,cno));
        currentset=repmat(false,length(previousset),1);
        currentset(rno)=true;
        if previousset==currentset
            rno=[rno; rno]; 
            oldval=[true;true]; 
            newval=[true;true];
        else
            rno=(find(abs(previousset-currentset))); 
            oldval=previousset(rno); newval=currentset(rno);                     
        end
        cno=repmat(cno,1,length(rno));
        runupdate=1;      
    end
    if checknext && strncmpi(T.datatype{cno},'button',3) 
        newval=get(gcbo,'String'); checknext=0; islegal=1; set(gcbo,'Value',0);        
    end
    if checknext && strncmpi(T.datatype{cno},'color',3)   
        checknext=0;
        if isnumeric(T.Data{rno,cno}) && numel(T.Data{rno,cno})==3 && max(T.Data{rno,cno})<=1
            oldval=T.Data{rno,cno};
        else
            oldval=[0.5 0.5 0.5];
        end
        newval=uisetcolor(oldval, 'Select Color');
    end
    
    if ~islegal runupdate=1; end
    
    % Updates T
    if (islegal && ~isequalwithequalnans(oldval,newval)) || strncmpi(T.datatype{cno(1)},'button',3) || strncmpi(T.datatype{cno(1)},'selection',3)
        if numel(rno)>1 
            for x=1:length(rno) T.Data{rno(x),cno(x)}=newval(x); end
        else
            T.Data{rno,cno}=newval;
        end
        T.lastedit_rowindex=rno;
        T.lastedit_colindex=cno;
        T.lastedit_datatype=T.datatype{cno(1)};
        T.lastedit_value=newval;
        T.lastedit_previousvalue=oldval;
        T.changedtime(rno,cno)=now;
        T.userinput(rno,cno)=1; 
        T.userinput(rno,cno)=1;
        set(bgrh,'UserData',T)
        runupdate=1;
    end
    
    % Updates table
    if runupdate  
        T=dispdata(T.TableName); 
    end
    % Runs callback
    if islegal && ~isequalwithequalnans(oldval,newval) || strncmpi(T.datatype{cno(1)},'button',3) || strncmpi(T.datatype{cno(1)},'selection',3)
        objh=get(gcf,'CurrentObject');
        eval(T.CellEditCallback); 
    end 
end

if call==4.2
    %disp('--- cellcallback new---')
    newval=0; islegal=1; checknext=1; runupdate=0; 
    T=get(gcbo,'UserData');
    rno=arg1.Indices(1); cno=arg1.Indices(2); inp=arg1.EditData;
    oldval=T.Data{rno,cno};
    
    if checknext && strncmpi(T.datatype{cno},'numeric',3)
        checknext=0;
        newval=str2double(inp);   
        [islegal,newval]=checkval(T,newval,rno,cno);  
    end
    if checknext && strncmpi(T.datatype{cno},'integer',3)
        checknext=0;
        newval=round(str2double(inp));   
        [islegal,newval]=checkval(T,newval,rno,cno);  
    end 
    if checknext && strncmpi(T.datatype{cno},'char',3)
        checknext=0;
        newval=inp;   
        [islegal,newval]=checkval(T,newval,rno,cno);  
    end 
    if checknext && strncmpi(T.datatype{cno},'selection',3)
         checknext=0;
        newval=inp; 
    end 
    if checknext && strncmpi(T.datatype{cno},'logical',3)
        checknext=0;
        newval=inp;
    end
    if checknext && strncmpi(T.datatype{cno},'excl_logical',3)
        checknext=0;
        previousset=cell2mat(T.Data(:,cno));
        currentset=repmat(false,length(previousset),1);
        currentset(rno)=true;
        rno=(find(abs(previousset-currentset)));
        newval=currentset(rno);
        oldval=previousset(rno);
        cno=repmat(cno,1,length(rno));
        runupdate=1;         
    end
    
    if ~islegal runupdate=1; end
    
    % Updates T
    if islegal && ~isequalwithequalnans(oldval,newval)
        if numel(rno)>1 
            for x=1:length(rno) T.Data{rno(x),cno(x)}=newval(x); end
        else
            T.Data{rno,cno}=newval;
        end
        T.lastedit_rowindex=rno;
        T.lastedit_colindex=cno;
        T.lastedit_datatype=T.datatype{cno(1)};
        T.lastedit_value=newval;
        T.lastedit_previousvalue=oldval;
        T.changedtime(rno,cno)=now;
        T.userinput(rno,cno)=1; 
        T.userinput(rno,cno)=1;
        set(gcbo,'UserData',T)
        runupdate=1;
    end
       
    % updates table
    if runupdate updatevals(T.TableName); end  

    % Runs callbackfunction
    if islegal && ~isequalwithequalnans(oldval,newval) 
        eval(T.CellEditCallback); 
    end
    
end

% Reports
if call>5 && call<6 userdat=get(gcbo,'Userdata'); tablename=userdat.tablename; end
if call==5.1 exporttable(tablename,'txt'); end
if call==5.2 exporttable(tablename,'xls'); end    
if call==5.3 exporttable(tablename,'dif'); end       
if call==5.4 exporttable(tablename,'slk'); end 
if call==5.5 exporttable(tablename,'csv'); end
if call==5.6 exporttable(tablename,'clip'); end
 
% Switch view
if call>6 && call<7 userdat=get(gcbo,'Userdata'); 
    tableh=userdat.tableh; 
    tablename=userdat.tablename;
    T=get(tableh,'Userdata');
    cleartable(tablename);
    if isfield(T,'O') T=rmfield(T,'O'); end
    if isfield(T,'N') T=rmfield(T,'N'); end
    if isfield(T,'L') T=rmfield(T,'L'); end   
end
if call==6.1
    %disp('--- switch to 1 ----')
    try T=initnew(T); catch T=initold(T); end  
end
if call==6.2
    %disp('--- switch to 2 ----')
    T=initold(T);
end
if call==6.3
    %disp('--- switch to 3 ----')
    T=initlist(T);
end

% List selection
if call==7.1
    %disp('--- listsel ---')
    T=get(gcbo,'UserData');
    T=listselection(T);
    if (~isfield(T,'ShowListSel') || T.ShowListSel) && (isfield(T,'ListSelCol') && T.ListSelCol)
        updatevals(T.TableName) % run only if listselection is used and visible
    end
end

% Select all / none
if call==7.2 || call==7.3   
    userdat=get(gcbo,'UserData');
    tableh=userdat.tableh; 
    tablename=userdat.tablename;
    T=get(tableh,'Userdata');   
    if call==7.2 val=true; else val=false; end
    [T.Data{:,T.ListSelCol}] = deal(val);
    set(tableh,'UserData',T)
    updatevals(tablename)
end

if call==90
    clearall
end

% Return control to calling object if object is in window
%if get(gco,'Parent') == gcf
%    objtype=get(gco,'Type');
%    switch objtype
%        case 'uitable'
%            uitable(gco)
%        case 'uicontrol'
%            uicontrol(gco)
%    end
%end


% -------------------------------------------------------------------------
% ----------------- S U B F U N C T I O N S -------------------------------
% -------------------------------------------------------------------------


%=========================================================================
%===========                 COMMON FUNCTIONS                 ============ 
%=========================================================================

% --------- CHECKVAL ---------
function [islegal,valout]=checkval(T,valin,rno,cno)

%disp('---- checkval ----')

islegal=1; %valout=valin;

if sum(strncmpi(T.datatype{cno},{'numeric','integer'},3))
    if valin < T.minvals(rno,cno) || valin > T.maxvals(rno,cno) islegal=0; end
    if isnan(valin) && ~T.allownans(rno,cno) islegal=0; end
    if isempty(valin) && ~T.allowempties(rno,cno) islegal=0; end
end

if strncmpi(T.datatype{cno},'char',3)
    if isempty(valin) && ~T.allowempties(rno,cno) islegal=0; end
end 

if ~islegal 
    valout=(T.Data{rno,cno});
else
    valout=valin;
end

if T.tabletype==2 && strcmpi(get(gcbo,'style'),'edit')
    if islegal set(gcbo,'BackgroundColor',[1 1 1]); else set(gcbo,'BackgroundColor',[1 1 0]); end
end
if T.tabletype==1 && ~islegal
    zz_message(1,'Illegal value','-r')
end

%disp('---- END checkval ----')

% -------- INITLIMITS ---------
function T=initlimits(T)

%disp('---- INITILIMITS ----')

% Sets minvals from MinVal & MinValCell
minval=-inf;
if isfield(T,'MinVal') && isnumeric(T.MinVal) minval=T.MinVal; end
minvals=repmat(minval,T.n_rows,T.n_cols);
if isfield(T,'MinVal') && iscell(T.MinVal) 
    for cno=1:length(T.MinVal)
        if ~isempty(T.MinVal{cno}) && isnumeric(T.MinVal{cno})
            [minvals(:,cno)]=deal(T.MinVal{cno});
        end
    end
end
if isfield(T,'MinValCell') && iscell(T.MinValCell)
    D=cellfun('isempty',T.MinValCell); 
    [i,j]=find(~D);
    for x=1:length(i)
        if isnumeric(T.MinValCell{i(x),j(x)}) && numel(T.MinValCell{i(x),j(x)})==1
            minvals(i(x),j(x))=T.MinValCell{i(x),j(x)};
        end
    end
end
T.minvals=minvals;

% Sets maxvals from MinVal & MinValCell
maxval=inf;
if isfield(T,'MaxVal') && isnumeric(T.MaxVal) maxval=T.MaxVal; end
maxvals=repmat(maxval,T.n_rows,T.n_cols);
if isfield(T,'MaxVal') && iscell(T.MaxVal) 
    for cno=1:length(T.MaxVal)
        if ~isempty(T.MaxVal{cno}) && isnumeric(T.MaxVal{cno})
            [maxvals(:,cno)]=deal(T.MaxVal{cno});
        end
    end
end
if isfield(T,'MaxValCell') && iscell(T.MaxValCell)
    D=cellfun('isempty',T.MaxValCell); 
    [i,j]=find(~D);
    for x=1:length(i)
        if isnumeric(T.MaxValCell{i(x),j(x)}) && numel(T.MaxValCell{i(x),j(x)})==1
            maxvals(i(x),j(x))=T.MaxValCell{i(x),j(x)};
        end
    end
end
T.maxvals=maxvals;

% Sets allownans from AllowNan & AllowNanCell
allow=0;
if isfield(T,'AllowNan') && isnumeric(T.AllowNan) && T.AllowNan>0 allow=1; end
allownans=repmat(allow,T.n_rows,T.n_cols);
if isfield(T,'AllowNan') && iscell(T.AllowNan) 
    for cno=1:length(T.AllowNan)
        if ~isempty(T.AllowNan{cno}) && isnumeric(T.AllowNan{cno}) && T.AllowNan{cno}>0
            [allownans(:,cno)]=deal(1);
        end
    end
end
if isfield(T,'AllowNanCell') && iscell(T.AllowNanCell)
    D=cellfun('isempty',T.AllowNanCell); 
    [i,j]=find(~D);
    for x=1:length(i)
        if isnumeric(T.AllowNanCell{i(x),j(x)}) && numel(T.AllowNanCell{i(x),j(x)})==1
            allownans(i(x),j(x))=T.AllowNanCell{i(x),j(x)}>0;
        end
    end
end
T.allownans=allownans;

% Sets allownans from AllowNan & AllowNanCell
allow=0;
if isfield(T,'AllowEmpty') && isnumeric(T.AllowEmpty) && T.AllowEmpty>0 allow=1; end
allowempties=repmat(allow,T.n_rows,T.n_cols);
if isfield(T,'AllowEmpty') && iscell(T.AllowEmpty) 
    for cno=1:length(T.AllowEmpty)
        if ~isempty(T.AllowEmpty{cno}) && isnumeric(T.AllowEmpty{cno}) && T.AllowEmpty{cno}>0
            [allowempties(:,cno)]=deal(1);
        end
    end
end
if isfield(T,'AllowEmptyCell') && iscell(T.AllowEmptyCell)
    D=cellfun('isempty',T.AllowEmptyCell); 
    [i,j]=find(~D);
    for x=1:length(i)
        if isnumeric(T.AllowEmptyCell{i(x),j(x)}) && numel(T.AllowEmptyCell{i(x),j(x)})==1
            allowempties(i(x),j(x))=T.AllowEmptyCell{i(x),j(x)}>0;
        end
    end
end
T.allowempties=allowempties;



% ----------- POPUPFIGURE -----------
function T=popupfigure(T)
global Zset
%disp('---- POPUPFIGURE ----')

if isfield(T,'PopupTitle') && ischar(T.PopupTitle) poptit=T.PopupTitle; else poptit=T.TableName; end

%winpar{1}=['PopupTable_' T.TableName]; winpar{2}=poptit; winpar{3}=T.Position;
%zz_drawwin(winpar,[],[])

wtag=['PopupTable_' T.TableName];
%winpos=T.Position;

wh=findobj('Tag',wtag);
if isempty(wh)
    
    wh=drawwin({wtag,poptit,T.Position});

    %---------------------- DEFAULT --------------
    set(wh,'DefaultUicontrolFontWeight','bold',...
        'DefaultUicontrolUnits','normalized',...
        'DefaultUicontrolFontSize',Zset.fonts(2),...
        'DefaultUicontrolBackgroundColor',get(gcbf,'Color'),...
        'DefaultUicontrolHorizontalAlignment','center')
    
    if isfield(T,'PopupModal') && T.PopupModal==1 set(wh,'WindowStyle','modal'); end
    if isfield(T,'PopupResizable') && T.PopupResizable==1 set(wh,'Resize','on'); end
    if isfield(T,'PopupCloseRequestFcn') && ischar(T.PopupCloseRequestFcn) set(wh,'CloseRequestFcn',T.PopupCloseRequestFcn); end  
    
    T.window_h=wh;
    T.window_position=T.Position;
    T.Position=[0.01 0.01 0.98 0.98];
end


% ----------- POPUPFIGURE -----------
function T=popupfigure_old(T)
global Zset
%disp('---- POPUPFIGURE ----')

if isfield(T,'PopupTitle') && ischar(T.PopupTitle) poptit=T.PopupTitle; else poptit=T.TableName; end

%winpar{1}=['PopupTable_' T.TableName]; winpar{2}=poptit; winpar{3}=T.Position;
%zz_drawwin(winpar,[],[])

wtag=['PopupTable_' T.TableName];
winpos=T.Position;

wh=findobj('Tag',wtag);
if isempty(wh)
    wh = figure('Color',get(gcbf,'Color'), ...
        'Units','normalized',...
        'Position',T.Position, ...
	    'Tag',wtag, ...
        'Name',poptit,...
        'NumberTitle','off',...
        'Handlevisibility','on',...
        'Menubar','none',...
        'ToolBar','none');

    %---------------------- DEFAULT --------------
    set(wh,'DefaultUicontrolFontWeight','bold',...
        'DefaultUicontrolUnits','normalized',...
        'DefaultUicontrolFontSize',Zset.fonts(2),...
        'DefaultUicontrolBackgroundColor',get(gcbf,'Color'),...
        'DefaultUicontrolHorizontalAlignment','center')
    
    if isfield(T,'PopupModal') && T.PopupModal==1 set(wh,'WindowStyle','modal'); end
    if isfield(T,'PopupResizable') && T.PopupResizable==1 set(wh,'Resize','on'); end
    if isfield(T,'PopupCloseRequestFcn') && ischar(T.PopupCloseRequestFcn) set(wh,'CloseRequestFcn',T.PopupCloseRequestFcn); end  
    
    T.window_h=wh;
    T.window_position=T.Position;
    T.Position=[0.01 0.01 0.98 0.98];
end


% ------------ INITEMPTY -----------
function T=initempty(T)
%disp('--- initempty ---')

T.tabletag=['TT_main_' T.TableName];

if isfield(T,'Popup') && T.Popup==1
    poph=findobj('Tag',['PopupTable_' T.TableName]); delete(poph)
    T=popupfigure(T); % Creates table in new window  
end

if ~isfield(T,'BackgroundColor') T.BackgroundColor=[1 1 1]; end
if ~isfield(T,'ForegroundColor') T.ForegroundColor=[0 0 0]; end 

T.window_h=gcf;

% Initialize positions
T.tableposition=T.Position;
WinUserData=get(T.window_h,'UserData');
winpos_norm=get(T.window_h,'Position');
pointwidth=WinUserData.pointwidth/winpos_norm(3);
pointheight=WinUserData.pointheight/winpos_norm(4);

if isfield(T,'FooterText')
    footerheight=numel(T.FooterText)*14*pointheight;
    T.tableposition(2)=T.tableposition(2)+footerheight;
    T.tableposition(4)=T.tableposition(4)-footerheight;
    T.footerposition=T.Position;
    T.footerposition(4)=footerheight;
end
if isfield(T,'HeaderText')
    headerheight=numel(T.HeaderText)*14*pointheight;
    T.tableposition(4)=T.tableposition(4)-headerheight;
    T.headerposition=T.Position;
    T.headerposition(2)=T.Position(2)+T.Position(4)-headerheight;
    T.headerposition(4)=headerheight;
end

th=findobj('Tag',['TT_main_' T.TableName]);
if isempty(th)  
   
    if isfield(T,'Popup') && T.Popup==1
        set(T.window_h,'Position',T.window_position);
    end

    % Create background
    th = uicontrol('Parent',T.window_h, ... 
        'Style','Frame', ...
        'BackgroundColor',[0.7 0.7 0.7], ...
        'ForegroundColor',[0.9 0.9 0.9],...
        'Position',T.tableposition, ...
        'Tag',['TT_main_' T.TableName]);
    
    % Sets message
    if isfield(T,'EmptyMessage')
        estr=T.EmptyMessage;
        if isempty(estr) estr=''; end
        if ischar(estr) estr={estr}; end
    else
        estr={'No Data'};
    end
    
    pos=T.tableposition;
    pos(2)=((2*pos(2)+pos(4))/2)-pointheight*7*(numel(estr)-1);
    pos(4)=14*pointheight*numel(estr);
    pos(1)=pos(1)+4*pointwidth;
    pos(3)=pos(3)-8*pointwidth;
    bgrh = uicontrol('Parent',T.window_h, ... 
        'Style','Text', ...
        'BackgroundColor',[0.7 0.7 0.7], ...
        'ForegroundColor',[0 0 0],...
        'Position',pos, ...
        'String',estr,...
        'Tag',['TT_text_' T.TableName]);    

    T.tabletype=0; % EMPTY TABLE
    
    if ~isfield(T,'FontSize')
        T.FontSize=10;
    end

    % Add header
    if isfield(T,'HeaderText') && isfield(T,'headerposition')
        T.header_h = uicontrol('Parent',T.window_h, ... 
            'Style','text', ...
            'FontSize',T.FontSize,...
            'FontWeight','Normal',...
            'HorizontalAlignment','left',...
            'BackgroundColor',get(T.window_h,'Color'), ...
            'Position',T.headerposition, ...	
            'String',T.HeaderText,...
            'Tag',['TT_header_' T.TableName]);
    end
    % Add footer
    if isfield(T,'FooterText') && isfield(T,'footerposition')    
        T.footer_h = uicontrol('Parent',T.window_h, ... 
            'Style','text', ...
            'FontSize',T.FontSize,...
            'FontWeight','Normal',...
            'HorizontalAlignment','left',...
            'BackgroundColor',get(T.window_h,'Color'), ...
            'Position',T.footerposition,...	
            'String',T.FooterText,...
            'Tag',['TT_header_' T.TableName]);
    end      
    
    set(th,'UserData',T)       
else
    zz_message(1,['Warning: Table ' T.TableName ' already exists'],'-r') 
end


% ------------ INITIALIZE -----------
function T=initialize(T)

%disp('---- INITIALIZE ----')

if ~iscell(T.Data) T.Data=num2cell(T.Data); end

T.tabletag=['TT_main_' T.TableName];

if isfield(T,'Popup') && T.Popup==1
    T=popupfigure(T); % Creates table in new window  
end

if ~isfield(T,'BackgroundColor') T.BackgroundColor=[1 1 1]; end
if ~isfield(T,'ForegroundColor') T.ForegroundColor=[0 0 0]; end 

T.window_h=gcf;

% Initialize positions
WinUserData=get(T.window_h,'UserData');
winpos_norm=get(T.window_h,'Position');
T.tableposition=T.Position;
pointheight=WinUserData.pointheight/winpos_norm(4);


if isfield(T,'FooterText')
    footerheight=numel(T.FooterText)*14*pointheight;
    T.tableposition(2)=T.tableposition(2)+footerheight;
    T.tableposition(4)=T.tableposition(4)-footerheight;
    T.footerposition=T.Position;
    T.footerposition(4)=footerheight;
end
if isfield(T,'HeaderText')
    headerheight=numel(T.HeaderText)*14*pointheight;
    T.tableposition(4)=T.tableposition(4)-headerheight;
    T.headerposition=T.Position;
    T.headerposition(2)=T.Position(2)+T.Position(4)-headerheight;
    T.headerposition(4)=headerheight;
end

[T.n_rows,T.n_cols]=size(T.Data);

T=initrownames(T); 
T=initcolnames(T); 
T=initformats(T); % Sets colum formats 
T=initcolwidths(T); % Set column widths
T=inteditables(T); % Sets editable columns
T=initlimits(T); % Sets max/min, NaNs, Empties

% Sets Font Size
if ~isfield(T,'FontSize')
    T.FontSize=10;
end

% Time settings
T.createdtime=now;
T.changedtime=repmat(now,T.n_rows,T.n_cols);
%T.userinput=repmat(0,T.n_rows,T.n_cols);
T.userinput=zeros(T.n_rows,T.n_cols);
%T.legalval=repmat(1,T.n_rows,T.n_cols);
T.legalval=ones(T.n_rows,T.n_cols);

% Initialize export values
T.lastedit_value=[];
T.lastedit_previousvalue=[];
T.lastedit_colindex=[];
T.lastedit_rowindex=[];


% ------------ EXPORT -------------
function exporttable(tablename,filetype)
global Zset
%disp('--- export ---')

if nargin==1; filetype='txt'; end

tableh=findobj('Tag',['TT_main_' tablename]);
T=get(tableh,'UserData');

% Sets rowtitles
for x=1:T.n_rows rowtitles{x}=['Row ' num2str(x)]; end; 
if isfield(T,'RowName') 
    if iscell(T.RowName)
        [rowtitles{:}]=deal(''); for x=1:length(T.RowName) rowtitles{x}=T.RowName{x}; end; 
    end
    if isempty(T.RowName) rowtitles=[]; end; 
end
for rno=1:length(rowtitles)
    Reportstruct(1).format{1,rno}='Str'; Reportstruct(1).cells{1,rno}=rowtitles{rno};
end
if isempty(rowtitles) coffs=0; else coffs=1; end

% Sets columntitles
for x=1:T.n_cols columntitles{x}=['Col ' num2str(x)]; end
if isfield(T,'ColumnName') 
    if iscell(T.ColumnName)
        [columntitles{:}]=deal(''); for x=1:length(T.ColumnName) columntitles{x}=T.ColumnName{x}; end
    end
    if isempty(T.ColumnName) columntitles=[]; end; 
end
for x=1:length(columntitles) Reportstruct(1).columntitle{x+coffs,1}=columntitles{x}; end

% Sets Data
for cno=1:T.n_cols
    if iscell(T.ColumnFormat{cno})
        for rno=1:T.n_rows
            Reportstruct(1).format{cno+coffs,rno}='Str'; Reportstruct(1).cells{cno+coffs,rno}=T.Data{rno,cno};
        end
    end
    if isnumeric(T.ColumnFormat{cno})
        for rno=1:T.n_rows
            Reportstruct(1).format{cno+coffs,rno}=T.ColumnFormat{cno}; Reportstruct(1).cells{cno+coffs,rno}=T.Data{rno,cno};
        end
    end
    if ischar(T.ColumnFormat{cno}) && strncmpi(T.ColumnFormat{cno},'numeric',3)
        for rno=1:T.n_rows
            Reportstruct(1).format{cno+coffs,rno}='%11.4g'; Reportstruct(1).cells{cno+coffs,rno}=T.Data{rno,cno};
        end
    end
    if ischar(T.ColumnFormat{cno}) && strncmpi(T.ColumnFormat{cno},'%',1)
         for rno=1:T.n_rows
            Reportstruct(1).format{cno+coffs,rno}=T.ColumnFormat{cno}; Reportstruct(1).cells{cno+coffs,rno}=T.Data{rno,cno};
        end
    end
    if ischar(T.ColumnFormat{cno}) && (strncmpi(T.ColumnFormat{cno},'string',3) || strncmpi(T.ColumnFormat{cno},'char',3) || strncmpi(T.ColumnFormat{cno},'text',3))
         for rno=1:T.n_rows
            Reportstruct(1).format{cno+coffs,rno}='Str'; Reportstruct(1).cells{cno+coffs,rno}=T.Data{rno,cno};
        end
    end
    if ischar(T.ColumnFormat{cno}) && sum(strncmpi(T.ColumnFormat{cno},{'logical','binary','excl_logical','excl_lbinary'},3))>0 
         for rno=1:T.n_rows
            Reportstruct(1).format{cno+coffs,rno}='Str'; 
            if T.Data{rno,cno}
                Reportstruct(1).cells{cno+coffs,rno}='true';
            else
                Reportstruct(1).cells{cno+coffs,rno}='false';
            end
        end
    end     
    if ischar(T.ColumnFormat{cno}) && strncmpi(T.ColumnFormat{cno},'color',3)        
        for rno=1:T.n_rows
            Reportstruct(1).format{cno+coffs,rno}='Str'; 
            if (numel(T.Data{rno,cno}))==3
                colr=T.Data{rno,cno}*255;
                colstr=[num2str(colr(1)) ':' num2str(colr(2)) ':' num2str(colr(3))];
            else
                colstr='-';
            end
            Reportstruct(1).cells{cno+coffs,rno}=colstr;       
        end
    end
end

% Header and footer
if isfield(T,'HeaderText')
    for x=1:numel(T.HeaderText)
        Reportstruct(1).header{x}=T.HeaderText{x}; 
    end
end
if isfield(T,'FooterText')
    for x=1:numel(T.FooterText)
        Reportstruct(1).footer{x}=T.FooterText{x}; 
    end
end

% Send to zz_report_xxx
if isfield(T,'ReportAddress') && ischar(T.ReportAddress) && ~isempty(T.ReportAddress)
    Reportstruct(1).pathname=T.ReportAddress;
else
    Reportstruct(1).pathname=[Zset.path_reports filesep 'Tableexport_' T.TableName];
end
if strcmpi(filetype,'txt')
    zz_report_txt(Reportstruct,Zset.report_aopen);
end
if strcmpi(filetype,'dif')
    zz_report_dif(Reportstruct,Zset.report_aopen);
end
if strcmpi(filetype,'xls')
    zz_report_xls_single(Reportstruct,Zset.report_aopen);
end
if strcmpi(filetype,'slk')
    zz_report_slk(Reportstruct,Zset.report_aopen);
end
if strcmpi(filetype,'csv')
    zz_report_csv(Reportstruct,Zset.report_aopen);
end
if strcmpi(filetype,'clip')
    zz_report_clip(Reportstruct,Zset);
end
% ------------ SETUIMENUS -------------
function T=setuimenus(tablename)
%disp('--- setuimenus ---')

tableh=findobj('Tag',['TT_main_' tablename]);
T=get(tableh,'UserData');

% Ensure figure with table is the current figure
tablepar=get(tableh,'Parent'); figure(tablepar)

userdat.tablename=tablename;
userdat.tableh=tableh;

tabletag=['TT_menu_'  tablename];

cmenu = uicontextmenu;
if T.tabletype==2 || T.tabletype==3
    uimenu(cmenu, 'Label', 'Table view 1', 'Callback', 'zz_tabletool(6.1);','Userdata',userdat);
end
if T.tabletype==1 || T.tabletype==3
    uimenu(cmenu, 'Label', 'Table view 2', 'Callback', 'zz_tabletool(6.2);','Userdata',userdat);
end
if T.tabletype==1 || T.tabletype==2
    uimenu(cmenu, 'Label', 'List view', 'Callback', 'zz_tabletool(6.3);','Userdata',userdat);
end
if isfield(T,'ListSelCol') && ~isempty(T.ListSelCol)
    if strcmp(T.datatype(T.ListSelCol),'logical') || strcmp(T.datatype(T.ListSelCol),'binary')
        uimenu(cmenu, 'Label', 'Select all', 'Callback', 'zz_tabletool(7.2);','separator','on','Userdata',userdat);
        uimenu(cmenu, 'Label', 'Select none', 'Callback', 'zz_tabletool(7.3);','Userdata',userdat); 
    end
end
uimenu(cmenu, 'Label', 'Export (txt)', 'Callback', 'zz_tabletool(5.1);','separator','on','Userdata',userdat);
uimenu(cmenu, 'Label', 'Export (xls)', 'Callback', 'zz_tabletool(5.2);','Userdata',userdat);
uimenu(cmenu, 'Label', 'Export (dif)', 'Callback', 'zz_tabletool(5.3);','Userdata',userdat);
uimenu(cmenu, 'Label', 'Export (slk)', 'Callback', 'zz_tabletool(5.4);','Userdata',userdat);
uimenu(cmenu, 'Label', 'Export (csv)', 'Callback', 'zz_tabletool(5.5);','Userdata',userdat);
uimenu(cmenu, 'Label', 'Copy (clipboard)', 'Callback', 'zz_tabletool(5.6);','Userdata',userdat);

set(cmenu,'Tag',tabletag)

set(tableh,'UiContextMenu',cmenu);

headerh=findobj('Tag',['TT_header_' T.TableName]);
if ~isempty(headerh) set(headerh,'UiContextMenu',cmenu); end
footerh=findobj('Tag',['TT_footer_' T.TableName]);
if ~isempty(footerh) set(footerh,'UiContextMenu',cmenu); end



% ------------ CLEARTABLE -------------
function cleartable(tablename)
%disp('--- cleartable ---')

delete(findobj('Tag',['TT_main_' tablename]))
delete(findobj('Tag',['TT_edit_' tablename]))
delete(findobj('Tag',['TT_rowname_' tablename]))
delete(findobj('Tag',['TT_colname_' tablename]))
delete(findobj('Tag',['TT_hslider_' tablename]))
delete(findobj('Tag',['TT_vslider_' tablename]))
delete(findobj('Tag',['TT_footer_' tablename]))
delete(findobj('Tag',['TT_header_' tablename]))
delete(findobj('Tag',['TT_text_' tablename]))
delete(findobj('Tag',['TT_menu_' tablename]))

% ------------ UPDATEPARAMETERS -------------
function T=updateparameter(T,param,value)
%disp('---- updateparameters ----')

tableh=findobj('Tag',['TT_main_' T.TableName]);

if T.tabletype==2
    switch param
        case 'ColumnName'
            T.(param)=value; T=initcolnames(T); set(tableh,'UserData',T);
            T=calcpos(T.TableName); T=setvisindex(T.TableName); T=visrlabels(T.TableName); T=visclabels(T.TableName); T=visedits(T.TableName);
            T=addcollabels(T.TableName);
        case 'RowName'
            T.(param)=value; T=initrownames(T); set(tableh,'UserData',T);
            T=calcpos(T.TableName); T=setvisindex(T.TableName); T=visrlabels(T.TableName); T=visclabels(T.TableName); T=visedits(T.TableName);
            T=addrowlabels(T.TableName);
        case 'ColumnWidth'
            T.(param)=value; T=initcolwidths(T); T=setcolumnwidths_old(T); set(tableh,'UserData',T);
            T=calcpos(T.TableName); T=setvisindex(T.TableName); T=visrlabels(T.TableName); T=visclabels(T.TableName); T=visedits(T.TableName);
        otherwise
            disp(['Setting for ' param ' not supported for table type 2'])
    end % Switch
end % if tabletype 2

if T.tabletype==1
    switch param
        case 'ColumnName'
            T.(param)=value; T=initcolnames(T); set(tableh,'ColumnName',T.ColumnName)
        case 'RowName'
            T.(param)=value; T=initrownames(T); set(tableh,'RowName',T.RowName)
        case 'ColumnWidth'
            T.(param)=value; T=initcolwidths(T); set(tableh,'ColumnWidth',T.ColumnWidth)
        otherwise
            disp(['Setting for ' param ' not supported for table type 1'])
    end
end

if T.tabletype==3
    switch param
        case 'ColumnName'
            T.(param)=value; T=initcolnames(T); set(tableh,'UserData',T); T=fillist(T.TableName); T=setlistselection(T.TableName);
        case 'RowName'
            T.(param)=value; T=initrownames(T); set(tableh,'UserData',T); T=fillist(T.TableName);
        otherwise
            disp(['Setting for ' param ' not supported for table type 1'])
    end % switch    
end
                      
set(tableh,'UserData',T) % Esures T is updated


% ------------ SETEDITABLES -----------
function T=inteditables(T)
%disp('---- inteditables ----')

editables=repmat(false,1,T.n_cols);
if isfield(T,'def_ColumnEditable'); 
    editables=repmat(T.def_ColumnEditable(1)>0,1,T.n_cols);
end
if isfield(T,'ColumnEditable') && iscell(T.ColumnEditable)
    for x=1:length(T.ColumnEditable)
        editables(x)=T.ColumnEditable{x}>0;
    end
end
if isfield(T,'ColumnEditable') && (isnumeric(T.ColumnEditable) || islogical(T.ColumnEditable))
    for x=1:length(T.ColumnEditable)
        editables(x)=T.ColumnEditable(x)>0;
    end
end
T.ColumnEditable=editables;


% ------------ INITCOLNAMES -----------
function T=initcolnames(T)
%disp('---- initcolnames ----')

if isfield(T,'ColumnName') && ~isempty(T.ColumnName)
    if strncmpi(T.ColumnName,{'numbered'},3) T.ColumnName='numbered'; end
    if iscell(T.ColumnName) 
        colnames=cell(1,T.n_cols); [colnames{:}]=deal('');
        setcols=find(~cellfun('isempty',T.ColumnName)); setcols(setcols>T.n_cols)=[];
        for x=1:length(setcols) colnames{setcols(x)}=T.ColumnName{setcols(x)}; end       
        T.ColumnName=colnames;    
    end
end


% ------------ SETROWNAMES -----------
function T=initrownames(T)
%disp('---- initrownames ----')

if ~isfield(T,'RowName') T.RowName=[]; end

if ~isempty(T.RowName)   
    if strncmpi(T.RowName,{'numbered'},3) T.RowName='numbered'; end
    if iscell(T.RowName) 
        rownames=cell(1,T.n_rows); [rownames{:}]=deal('');
        setrows=find(~cellfun('isempty',T.RowName)); setrows(setrows>T.n_rows)=[];
        for x=1:length(setrows) rownames{setrows(x)}=T.RowName{setrows(x)}; end       
        T.RowName=rownames;
    end
end


% ------------ INITFORMATS -----------
function T=initformats(T)
%disp('---- initformats ----')

% Sets column formats
formats=cell(1,T.n_cols); [formats{:}]=deal('char');
if isfield(T,'def_ColumnFormat') 
    [formats{:}]=deal(T.def_ColumnFormat);
end
if isfield(T,'ColumnFormat') && ~isempty(T.ColumnFormat)
    for x=1:length(T.ColumnFormat)
        if ~isempty(T.ColumnFormat{x})
            formats{x}=T.ColumnFormat{x};
        end
    end
end
T.ColumnFormat=formats;

% sets datatype
for cno=1:length(T.ColumnFormat) 
    checknext=1; 
    if checknext && iscell(T.ColumnFormat{cno}) T.datatype{cno}='selection'; checknext=0; end
    if checknext && isnumeric(T.ColumnFormat{cno}) T.datatype{cno}='numeric'; checknext=0; end
    if checknext && sum(strcmpi(T.ColumnFormat{cno},{'logical','binary'})) T.datatype{cno}='logical'; checknext=0; end
    if checknext && sum(strcmpi(T.ColumnFormat{cno},{'excl_logical','excl_binary'})) T.datatype{cno}='excl_logical'; checknext=0; end
    if checknext && strncmpi(T.ColumnFormat{cno},'%',1) T.datatype{cno}='numeric'; checknext=0; end
    if checknext && strncmpi(T.ColumnFormat{cno},'int',3) T.datatype{cno}='integer'; checknext=0; end 
    if checknext && sum(strncmpi(T.ColumnFormat{cno},{'char','string','text'},3)) T.datatype{cno}='char'; checknext=0; end
    if checknext && strncmpi(T.ColumnFormat{cno},'numeric',3) T.datatype{cno}='numeric'; checknext=0; end    
    if checknext && strncmpi(T.ColumnFormat{cno},'color',3) T.datatype{cno}='color'; checknext=0; end  
    if checknext && strncmpi(T.ColumnFormat{cno},'button',3) T.datatype{cno}='button'; checknext=0; end  
end

% Converts binary data to logical
for cno=1:length(T.ColumnFormat) 
    if ischar(T.ColumnFormat{cno}) && sum(strcmpi(T.ColumnFormat{cno},{'logical','binary','excl_logical','excl_binary'}))
        for rno=1:T.n_rows
            if T.Data{rno,cno}>0 T.Data{rno,cno}=true; else T.Data{rno,cno}=false; end
        end
    end
end


% ------------ SETCOLWIDTHS -----------
function T=initcolwidths(T)
%disp('---- initcolwidths ----')
% Sets column widths
cwidths=cell(1,T.n_cols); [cwidths{:}]=deal('auto');
if isfield(T,'def_ColumnWidth') && isnumeric(T.def_ColumnWidth) && ~isempty(T.def_ColumnWidth)
    [cwidths{:}]=deal(T.def_ColumnWidth);
end
if isfield(T,'ColumnWidth') && ~isempty(T.ColumnWidth) && iscell(T.ColumnWidth)
    for x=1:length(T.ColumnWidth)
        if ~isempty(T.ColumnWidth{x})
            cwidths{x}=T.ColumnWidth{x};
        end
    end   
end
T.ColumnWidth=cwidths;


% ------------ UPDATEVALS -----------
function updatevals(tablename)

%disp('--- updatevals ----')

tableh=findobj('Tag',['TT_main_' tablename]);
T=get(tableh,'UserData');

if T.tabletype==1
    T=dat2str_new(tablename);
    set(tableh,'Data',T.N.strdata)
end
if T.tabletype==2
    dispdata(tablename);
end
if T.tabletype==3
    T=fillist(tablename);
    T=setlistselection(tablename);
end

%disp('--- end updatevals ----')

    
%=========================================================================
%===========          FUNCTIONS FOR NEW STYLE TABLES           =========== 
%========================================================================= 
   

% ------------ INITNEW -----------
function T=initnew(T)
%disp('--- initnew ---')

th=findobj('Tag',['TT_main_' T.TableName]);

if isempty(th)   
        
    % Create table
    th=uitable('Units','normalized',...
        'Position',T.tableposition,...
        'Tag',['TT_main_' T.TableName],...
        'UserData',T);
    
    T=dat2str_new(T.TableName);
    set(th,'Data',T.N.strdata);
    
    set(th,'BackgroundColor',T.BackgroundColor)
    set(th,'ForegroundColor',T.ForegroundColor)
    set(th,'CellEditCallback',@cellcallnew) 
    set(th,'ColumnEditable',T.N.coleditable)
    set(th,'ColumnFormat',T.N.newtabcolformat)
    set(th,'ColumnWidth',T.ColumnWidth)
    set(th,'FontSize',T.FontSize)
    
    T.tabletype=1;

    % Add header
    if isfield(T,'HeaderText') && isfield(T,'headerposition')
        T.header_h = uicontrol('Parent',T.window_h, ... 
            'Style','text', ...
            'FontSize',T.FontSize,...
            'FontWeight','Normal',...
            'HorizontalAlignment','left',...
            'BackgroundColor',get(T.window_h,'Color'), ...
            'Position',T.headerposition, ...	
            'String',T.HeaderText,...
            'Tag',['TT_header_' T.TableName]);
    end
    % Add footer
    if isfield(T,'FooterText') && isfield(T,'footerposition')    
        T.footer_h = uicontrol('Parent',T.window_h, ... 
            'Style','text', ...
            'FontSize',T.FontSize,...
            'FontWeight','Normal',...
            'HorizontalAlignment','left',...
            'BackgroundColor',get(T.window_h,'Color'), ...
            'Position',T.footerposition,...	
            'String',T.FooterText,...
            'Tag',['TT_header_' T.TableName]);
    end      

    % Row labels, column labels, uimenu
    if isfield(T,'ColumnName') set(th,'ColumnName',T.ColumnName); end
    if isfield(T,'RowName') set(th,'RowName',T.RowName); end

    set(th,'UserData',T)
    
    T=setuimenus(T.TableName);

      
else
    zz_message(1,['Warning: Table ' T.TableName ' already exists'],'-r') 
end


% --------- CELLCALLNEW --------
function cellcallnew(fh,arg)
%disp('--- cellcallnew ---')
% used to pass cell contents from new table

zz_tabletool(4.2,arg);

% ----------- dat2str_new -----------%
function T=dat2str_new(tablename)
%disp('--- dat2str_new ---')

bgrh=findobj('Tag',['TT_main_' tablename]);
T=get(bgrh,'UserData');

[n_rows,n_cols]=size(T.Data);
T.N.strdata=T.Data;
T.N.coleditable=T.ColumnEditable;

% Create stringoutput
for cno=1:length(T.ColumnFormat) 
    checknext=1; 
    if checknext && iscell(T.ColumnFormat{cno})
        T.N.newtabcolformat{cno}=T.ColumnFormat{cno}; 
        checknext=0;    
    end
    if checknext && isnumeric(T.ColumnFormat{cno})
        T.N.newtabcolformat{cno}='char'; 
        checknext=0;
        for rno=1:n_rows
            T.N.strdata{rno,cno}=num2str(T.Data{rno,cno},T.ColumnFormat{cno});
        end       
    end
    if checknext && (strcmpi(T.ColumnFormat{cno},'logical') || strcmpi(T.ColumnFormat{cno},'binary'))
        T.N.newtabcolformat{cno}='logical'; 
        checknext=0;
        for rno=1:n_rows
            T.N.strdata{rno,cno}=T.Data{rno,cno}>0;
        end
    end
    if checknext && (strcmpi(T.ColumnFormat{cno},'excl_logical') || strcmpi(T.ColumnFormat{cno},'excl_binary'))
        T.N.newtabcolformat{cno}='logical'; 
        checknext=0;
        for rno=1:n_rows
            T.N.strdata{rno,cno}=T.Data{rno,cno}>0;
        end
    end
    if checknext && strncmpi(T.ColumnFormat{cno},'%',1)
        T.N.newtabcolformat{cno}='char'; 
        checknext=0;
        for rno=1:n_rows
            T.N.strdata{rno,cno}=num2str(T.Data{rno,cno},T.ColumnFormat{cno});
        end        
    end
    if checknext && strncmpi(T.ColumnFormat{cno},'int',3)
        T.N.newtabcolformat{cno}='char'; 
        checknext=0;
        for rno=1:n_rows
            T.N.strdata{rno,cno}=num2str(round(T.Data{rno,cno}));
        end        
    end    
    if checknext && (strncmpi(T.ColumnFormat{cno},'char',3) || strncmpi(T.ColumnFormat{cno},'string',3)) 
        T.N.newtabcolformat{cno}='char';
        checknext=0;
        for rno=1:n_rows
            T.N.strdata{rno,cno}=num2str(T.Data{rno,cno});
        end 
    end
    if checknext && strncmpi(T.ColumnFormat{cno},'text',3) 
        T.N.newtabcolformat{cno}='char';
        T.N.coleditable(cno)=0;
        checknext=0;
        for rno=1:n_rows
            T.N.strdata{rno,cno}=num2str(T.Data{rno,cno});
        end 
    end
    if checknext && strncmpi(T.ColumnFormat{cno},'button',3) 
        T.N.newtabcolformat{cno}='char';
        T.N.coleditable(cno)=0;
        checknext=0;
        for rno=1:n_rows
            T.N.strdata{rno,cno}=['[' num2str(T.Data{rno,cno}) ']'];
        end 
    end    
    if checknext && strncmpi(T.ColumnFormat{cno},'numeric',3)   
        T.N.newtabcolformat{cno}='char';
        for rno=1:n_rows
            T.N.strdata{rno,cno}=num2str(T.Data{rno,cno});
        end 
    end
    if checknext && strncmpi(T.ColumnFormat{cno},'color',3)   
        T.N.newtabcolformat{cno}='char';
        T.N.coleditable(cno)=0;
        for rno=1:n_rows
            colr=T.Data{rno,cno}*255;       
            if isnumeric(T.Data{rno,cno}) && numel(T.Data{rno,cno})==3
                T.N.strdata{rno,cno}=[num2str(colr(1)) ':' num2str(colr(2)) ':' num2str(colr(3))];
            else           
                T.N.strdata{rno,cno}='-';
            end
        end 
    end       
end


% Sets colors for non-editable cells
ind=find(~T.N.coleditable);
for x=1:length(ind)    
    if ~strncmpi(T.datatype{ind(x)},{'selection','logical','excl_logical'},3)
        for rno=1:n_rows     
            T.N.strdata{rno,ind(x)}=['<HTML><font color=#0000FF>' T.N.strdata{rno,ind(x)} '</TD>'];
        end
    end
end


%=========================================================================
%===========           FUNCTIONS FOR OLDSTYLE TABLES           =========== 
%=========================================================================

% ------------ INITOLD -----------
function T=initold(T)
%disp('--- initold ---')

bgrh=findobj('Tag',['TT_main_' T.TableName]);
if isempty(bgrh)
    
    if isfield(T,'Popup') && T.Popup==1
        set(T.window_h,'Position',T.window_position);
    end

    % Create background
    bgrh = uicontrol('Parent',T.window_h, ... 
        'Style','Frame', ...
        'BackgroundColor',[0.7 0.7 0.7], ...
        'ForegroundColor',[0.9 0.9 0.9],...
        'Position',T.Position, ...	
        'Tag',['TT_main_' T.TableName]);

    pos=T.tableposition;

    T.O.pos_top=pos(2)+pos(4);
    T.O.pos_bottom=pos(2);
    T.O.pos_left=pos(1);
    T.O.pos_right=pos(1)+pos(3);

    WinUserData=get(T.window_h,'UserData');
    winpos_norm=get(T.window_h,'Position');
    T.O.pointheight=WinUserData.pointheight/winpos_norm(4);
    T.O.pointwidth=WinUserData.pointwidth/winpos_norm(3);
    T.O.pixelheight=WinUserData.pixheight/winpos_norm(4);
    T.O.pixelwidth=WinUserData.pixwidth/winpos_norm(3);   
    
    T=setcolumnwidths_old(T);   
    T.tabletype=2;
    set(bgrh,'UserData',T) % Adds T to Userdata before routines below

    % draws table
    T=setuimenus(T.TableName); % Add menus
    T=calcpos(T.TableName);    
    T=headerfooter(T.TableName); % Add header/footer
    T=addrowlab(T.TableName); % Add row labels
    T=addobj(T.TableName); % add objects
    T=dispdata(T.TableName); % Display data
    
   
else
    zz_message(1,['Warning: Table ' T.TableName ' already exists'],'-r')
end


% ----------- setcolumnwidths_old -----------%
function T=setcolumnwidths_old(T)
%disp('--- setcolumnwidths_old ---')

% Sets column widths   
defwidth=100;
if isfield(T,'def_ColumnWidth') && isnumeric(T.def_ColumnWidth) && ~isempty(T.def_ColumnWidth)
    defwidth=T.def_ColumnWidth(1);
end
T.O.columnwidths_pix=repmat(defwidth,1,T.n_cols);
if isfield(T,'ColumnWidth') && iscell(T.ColumnWidth)
    for x=1:length(T.ColumnWidth)
        if isnumeric(T.ColumnWidth{x})
            T.O.columnwidths_pix(x)=T.ColumnWidth{x};
        end
    end
end
T.O.columnwidths=T.O.columnwidths_pix*T.O.pixelwidth;


% ---------------- DISPDATA --------------
function T=dispdata(tablename)

%disp('--- zz_tabletool_dispdata ---')

bgrh=findobj('Tag',['TT_main_' tablename]);
T=get(bgrh,'UserData');

%nrows=min([T.O.n_visrows T.O.n_rows]);
%ncols=numel(T.O.collab_h);
[nrows,ncols]=size(T.O.cell_h);

% display rownames
if T.O.rnames_w>0
    % Numeric
    if ischar(T.RowName) && strncmpi(T.RowName,'numbered',3)
        for rno=1:nrows
            rindex=rno+T.O.firstrowindex-1;
            set(T.O.rowlab_h(rno),'String',num2str(rindex),'HorizontalAlignment','left',...
                'TooltipString',num2str(rindex))
        end
    else % Text      
        for rno=1:nrows
            rindex=rno+T.O.firstrowindex-1;
            set(T.O.rowlab_h(rno),'String',T.RowName{rindex},'HorizontalAlignment','left',...
                'TooltipString',T.RowName{rindex})
        end
    end
end

% Display cell content
roffs=T.O.firstrowindex-1;
for cno=1:ncols
    cindex=cno+T.O.firstcolindex-1; % index to T.data
    checknext=1;
    
    % Popupmenu (string set in addobj)
    if checknext && iscell(T.ColumnFormat{cindex})
        checknext=0;    
        for rno=1:nrows
            cstr=get(T.O.cell_h(rno,cno),'String');
            mstr=T.Data{rno+roffs,cindex};
            ind=find(strcmp(mstr,cstr));
            if ~isempty(ind)
                set(T.O.cell_h(rno,cno),'Value',ind(1))
            end   
        end       
    end
    % Numeric unspecified
    if checknext && strncmpi(T.ColumnFormat{cindex},'numeric',3)   
        for rno=1:nrows
            set(T.O.cell_h(rno,cno),'String',num2str(T.Data{rno+roffs,cindex}));
        end 
    end   
    % Numeric, sign digits specified
    if checknext && isnumeric(T.ColumnFormat{cindex})
        checknext=0;
        for rno=1:nrows           
            set(T.O.cell_h(rno,cno),'String',num2str(T.Data{rno+roffs,cindex},T.ColumnFormat{cindex}));
        end       
    end
    % Numeric, specified format
    if checknext && strncmpi(T.ColumnFormat{cindex},'%',1)
        checknext=0;
        for rno=1:nrows
            set(T.O.cell_h(rno,cno),'String',num2str(T.Data{rno+roffs,cindex},T.ColumnFormat{cindex}));
        end        
    end
    % integer
    if checknext && strncmpi(T.ColumnFormat{cindex},'int',3)
        checknext=0;
        for rno=1:nrows
            set(T.O.cell_h(rno,cno),'String',num2str(round(T.Data{rno+roffs,cindex})));
        end        
    end 
    % Binary
    if checknext && sum(strcmpi(T.ColumnFormat{cindex},{'logical','binary','excl_logical','excl_binary'})) 
        checknext=0;
        for rno=1:nrows
            set(T.O.cell_h(rno,cno),'Value',T.Data{rno+roffs,cindex})
        end
    end
    % text
    if checknext && (strncmpi(T.ColumnFormat{cindex},'text',3))
        checknext=0;
        for rno=1:nrows
            set(T.O.cell_h(rno,cno),'String',[' ' num2str(T.Data{rno+roffs,cindex})],'TooltipString',[' ' num2str(T.Data{rno+roffs,cindex})]);
        end 
    end
    % String
    if checknext && (strncmpi(T.ColumnFormat{cindex},'char',3) || strncmpi(T.ColumnFormat{cindex},'string',3) || strncmpi(T.ColumnFormat{cindex},'pushbuton',3))   
        checknext=0;
        for rno=1:nrows
            set(T.O.cell_h(rno,cno),'String',num2str(T.Data{rno+roffs,cindex}));
        end 
    end 
    % Color
    if checknext && strncmpi(T.ColumnFormat{cindex},'color',3) 
        %checknext=0; % Activate if more options
        for rno=1:nrows
            if isnumeric(T.Data{rno+roffs,cindex}) && numel(T.Data{rno+roffs,cindex})==3
                set(T.O.cell_h(rno,cno),'BackgroundColor',T.Data{rno+roffs,cindex})
            end
        end        
    end
    
end

set(bgrh,'UserData',T)

%disp('--- end dispdata ---')

% ------------ FUNCTION CALCPOS ----------
function T=calcpos(tablename)
% Calculates required widths and heights
% Adds sliders if required
%disp('---- calcpos ----')

bgrh=findobj('Tag',['TT_main_' tablename]);
T=get(bgrh,'UserData');

T.O.fho=T.O.pixelwidth*2; % frame horizontal offset
T.O.fvo=T.O.pixelheight*1; % frame vertical offset

[T.O.n_rows,T.O.n_cols]=size(T.Data);

T.O.edit_h=T.FontSize*T.O.pointheight*1.9;

% Sets row names w.
T.O.rnames_w=T.O.pixelwidth*100;
if isfield(T,'RowNamesWidth') && isnumeric(T.RowNamesWidth) && ~isempty(T.RowNamesWidth)
    T.O.rnames_w=T.O.pixelwidth*T.RowNamesWidth;
end
if isfield(T,'RowName') && isempty(T.RowName)
    T.O.rnames_w=0;
end

T.O.cnames_h=T.O.edit_h;
% sets col names h.
if isfield(T,'ColumnNamesHeight') && isnumeric(T.ColumnNamesHeight) && ~isempty(T.ColumnNamesHeight)
    T.O.cnames_h=T.O.pointheight*T.ColumnNamesHeight;
end
if isfield(T,'ColumnName') && isempty(T.ColumnName)
    T.O.cnames_h=0;
end    

% Sets condition for v-slider and hbreak
sliderw=12*T.O.pointwidth;
sliderh=12*T.O.pointheight;
vslider=0; hslider=0;

%Calculates required width and heights+breakpoints
hbreak=T.O.pos_right-T.O.fho; 
vbreak=T.O.pos_bottom+T.O.fvo; 
available_w=T.O.pos_right-T.O.pos_left-T.O.rnames_w-T.O.fho*2;
available_h=T.O.pos_top-T.O.pos_bottom-T.O.cnames_h-T.O.fvo*2; 

% Checks if vslider should be added, updated requiredw
if (T.O.n_rows*T.O.edit_h)>available_h
    vslider=1; 
    available_w=available_w-sliderw;
    hbreak=T.O.pos_right-T.O.fho-sliderw;
end
if sum(T.O.columnwidths)>available_w
    hslider=1;
    available_h=available_h-sliderh;
    vbreak=T.O.pos_bottom+T.O.fvo+sliderh;
end
% New test
if (T.O.n_rows*T.O.edit_h)>available_h
    vslider=1;
    hbreak=T.O.pos_right-T.O.fho-sliderw;
end
% New test
if sum(T.O.columnwidths)>available_w
    hslider=1;
    vbreak=T.O.pos_bottom+T.O.fvo+sliderh;
end

%T.O.edit_h=edit_h;
T.O.listtop=T.O.pos_top-T.O.fvo-T.O.cnames_h;
T.O.listbottom=vbreak;
T.O.listleft=T.O.pos_left+T.O.fho+T.O.rnames_w;
T.O.listright=hbreak;
T.O.listwidth=T.O.listright-T.O.listleft;
T.O.rownamesleft=T.O.pos_left+T.O.fho;
T.O.rownameswidth=T.O.rnames_w;
T.O.colnamesbottom=T.O.pos_top-T.O.fvo-T.O.cnames_h;
T.O.colnamesheight=T.O.cnames_h;
T.O.n_visrows=floor((T.O.listtop-T.O.listbottom)/(T.O.edit_h+T.O.fvo));

% Check for columns wider than table
maxcolwidth=T.O.listwidth;
toowide=T.O.columnwidths>maxcolwidth;
T.O.columnwidths(toowide)=maxcolwidth*0.99;

% Column and row indices
T.O.firstrowindex=1; 
T.O.firstcolindex=1;
T.O.maxfirstrowindex=max([T.O.n_rows-T.O.n_visrows+1 1]);

rcumwidths=cumsum(fliplr(T.O.columnwidths+T.O.fho)); % Reverse comulative column widths
rncol=sum(rcumwidths<=T.O.listwidth); 
T.O.maxfirstcolindex=max([T.O.n_cols-rncol+1 1]);

% ADD SLIDERS 	
vsh=[]; hsh=[];
if vslider && isempty(findobj('Tag',['TT_vslider_' T.TableName]))
    %disp('---- added vslider ---')
    sst=1/(T.O.n_rows-T.O.n_visrows); % small step
    lst=T.O.n_visrows/T.O.n_rows; % large  step
    vsh = uicontrol('Parent',T.window_h, ... 
        'Style','Slider', ...
        'Position',[hbreak vbreak sliderw T.O.listtop-T.O.listbottom], ...	
        'Tag',['TT_vslider_' T.TableName],...
        'Callback','zz_tabletool(3.1);',...
        'Value',1,...
        'SliderStep',[sst lst]); 
end

if hslider && isempty(findobj('Tag',['TT_hslider_' T.TableName]))
    %disp('---- added hslider ----')
    hsh = uicontrol('Parent',T.window_h, ... 
        'Style','Slider', ...
        'Position',[T.O.listleft T.O.pos_bottom+T.O.fvo T.O.listright-T.O.listleft sliderh], ...	
        'Tag',['TT_hslider_' T.TableName],...
        'Callback','zz_tabletool(3.2);');   
    % Sliderstep adjusted in addobj
end

userdat.tablename=T.TableName; 
userdat.tablehandle=bgrh;
userdat.tabletag=get(bgrh,'Tag');
set([vsh hsh],'UserData',userdat,'BackgroundColor',[0.9 0.9 0.9])

set(bgrh,'UserData',T)


% ---------------- ADDROWLAB -------------
function T=addrowlab(tablename)

bgrh=findobj('Tag',['TT_main_' tablename]); % Handle to background
T=get(bgrh,'UserData'); % gets T

cmenu=get(bgrh,'UiContextMenu');

% Deletes existing
delobj=findobj(gcf,'Tag',['TT_rowname_' T.TableName]); delete(delobj)

% Add row labels
if T.O.rnames_w>0
    
    % set number of rows
    [n_rows,n_cols]=size(T.Data);
    n_rows=min([n_rows,T.O.n_visrows]);
    
    handoffs=T.O.edit_h+T.O.fvo; %Height and offset

    opos=[T.O.pos_left+T.O.fho T.O.listtop-handoffs T.O.rnames_w T.O.edit_h]; % object position
    for rno=1:n_rows
        T.O.rowlab_h(rno) = uicontrol('Parent',T.window_h, ... 
            'Style','text', ...
            'FontSize',T.FontSize,...
            'BackgroundColor',[0.9 0.9 0.9], ...
            'Position',opos, ...	
            'String',num2str(rno),...
            'Visible','on',...
            'Enable','on',...
            'FontWeight','bold',...
            'HorizontalAlignment','center',...
            'UiContextMenu',cmenu,...
            'Tag',['TT_rowname_' T.TableName]);  
       opos(2)=opos(2)-handoffs; 
    end
end

set(bgrh,'UserData',T)



% ---------------- ADDOBJ -------------
function T=addobj(tablename)
%disp('---- addobj ----')

bgrh=findobj('Tag',['TT_main_' tablename]); % Handle to background
T=get(bgrh,'UserData'); % gets T

cmenu=get(bgrh,'UiContextMenu');

% Deletes existing
T.O.cell_h=[]; T.O.collab_h=[];
delobj=findobj(gcf,'Tag',['TT_edit_' T.TableName]); delete(delobj)
delobj=findobj(gcf,'Tag',['TT_colname_' T.TableName]); delete(delobj)

handoffs=T.O.edit_h+T.O.fvo; %Height and offset

% Calculate column widths
colwidths=T.O.columnwidths;
if T.O.firstcolindex>1
    colwidths(1:T.O.firstcolindex-1)=[];
end
colwidths=colwidths+T.O.fho;
cumwidths=cumsum(colwidths);
ncol=sum(cumwidths<=T.O.listwidth);

% Correct widths
if isfield(T,'HorizontalExpansion') && T.HorizontalExpansion>0
    cf=T.O.listwidth/(cumwidths(ncol)+ncol*T.O.fho);
    colwidths=(colwidths*cf);
end

% set number of rows
[n_rows,n_cols]=size(T.Data);
n_rows=min([n_rows,T.O.n_visrows]);

opos(1)=T.O.listleft; %+T.O.fho; 
opos(4)=T.O.edit_h;
for cno=1:ncol
    
    colindex=cno+T.O.firstcolindex-1; % Index to original table
    
    % Positions
    opos(2)=T.O.listtop-handoffs;
    opos(3)=colwidths(cno);   
    
    % ADD OBJECT TYPE
    ButtonDownFcn=''; % Unused?
    if ischar(T.ColumnFormat{colindex}) && sum(strncmpi(T.ColumnFormat{colindex},{'char','string','text','button'},3)) alignment='left'; else alignment='right'; end
    if T.ColumnEditable(colindex) enablestr='on'; else enablestr='off'; end
    if ischar(T.ColumnFormat{colindex}) && strncmpi(T.ColumnFormat{colindex},'text',3) enablestr='on'; end
    % Set types for all and string for selections
    stylestr='edit'; strstr=''; bgcolor=[1 1 1];
    if strncmpi(T.ColumnFormat{colindex},{'text'},3) stylestr='text';  bgcolor=[0.9 0.9 0.9]; end
    if strncmpi(T.ColumnFormat{colindex},{'color'},3) stylestr='pushbutton'; bgcolor=[0.9 0.9 0.9]; end
    if strncmpi(T.ColumnFormat{colindex},{'button'},3) stylestr='togglebutton'; bgcolor=[0.9 0.9 0.9]; end
    if iscell(T.ColumnFormat{colindex})
        stylestr='popupmenu'; strstr=T.ColumnFormat{colindex};
    else
        if sum(strncmpi(T.ColumnFormat{colindex},{'logical','excl_logical','binary','excl_binary'},3)) stylestr='checkbox'; bgcolor=[0.9 0.9 0.9]; end 
    end
     
    % userdata
    userdat.tablename=T.TableName; 
    userdat.tablehandle=bgrh;
    userdat.tabletag=get(bgrh,'Tag');
    userdat.edit_cno=cno;     
      
    for rno=1:n_rows
        userdat.edit_rno=rno;  % userdata row number
        T.O.cell_h(rno,cno) = uicontrol('Parent',T.window_h, ... 
            'Style',stylestr, ...
            'FontSize',T.FontSize,...
            'BackgroundColor',bgcolor, ...
            'Position',opos, ...	
            'String',strstr,...
            'Visible','on',...
            'Enable',enablestr,...
            'FontWeight','bold',...
            'HorizontalAlignment',alignment,...
            'Tag',['TT_edit_' T.TableName],...
            'CallBack','zz_tabletool(4.1);',...  
            'ButtonDownFcn',ButtonDownFcn,...
            'UiContextMenu',cmenu,...
            'Userdata',userdat);  
       opos(2)=opos(2)-handoffs; 
    end
       
    % Add column headers
    if T.O.colnamesheight>0
        T.O.collab_h(cno) = uicontrol('Parent',T.window_h, ... 
            'Style','text', ...
            'FontSize',T.FontSize,...
            'BackgroundColor',[0.9 0.9 0.9], ...
            'Position',[opos(1) T.O.colnamesbottom opos(3) T.O.colnamesheight], ...	
            'String',T.ColumnName{colindex},...
            'TooltipString',T.ColumnName{colindex},...
            'Visible','on',...
            'Enable','on',...
            'FontWeight','bold',...
            'HorizontalAlignment','center',...
            'UiContextMenu',cmenu,...
            'Tag',['TT_colname_' T.TableName]);  
    end
    opos(1)=opos(1)+colwidths(cno); %+T.O.fho;
          
end

% adjust horizontal slider
if ncol<T.O.n_cols
    sst=1/(T.O.n_cols-ncol);
    lst=ncol/T.O.n_cols;
    if sst>lst sst=lst; end
    sh=findobj(gcf,'Tag',['TT_hslider_' T.TableName]);    
    set(sh,'SliderStep',[sst lst])    
end

set(bgrh,'UserData',T)



% ---------------- HEADERFOOTER --------------
function T=headerfooter(tablename)

bgrh=findobj('Tag',['TT_main_' tablename]);
T=get(bgrh,'UserData');

% Delete existing
delobj=findobj(gcf,'Tag',['TT_header_' T.TableName]); delete(delobj)

% Add header
if isfield(T,'HeaderText') && isfield(T,'headerposition')
    T.header_h = uicontrol('Parent',T.window_h, ... 
        'Style','text', ...
        'FontSize',T.FontSize,...
        'FontWeight','Normal',...
        'HorizontalAlignment','left',...
        'BackgroundColor',get(T.window_h,'Color'), ...
        'Position',T.headerposition, ...	
        'String',T.HeaderText,...
        'Tag',['TT_header_' T.TableName]);
end
% Add footer
if isfield(T,'FooterText') && isfield(T,'footerposition')    
    T.footer_h = uicontrol('Parent',T.window_h, ... 
        'Style','text', ...
        'FontSize',T.FontSize,...
        'FontWeight','Normal',...
        'HorizontalAlignment','left',...
        'BackgroundColor',get(T.window_h,'Color'), ...
        'Position',T.footerposition,...	
        'String',T.FooterText,...
        'Tag',['TT_header_' T.TableName]); % TEMP Tag should possibly be footer, check handling elsewhere
end 

set(bgrh,'UserData',T)




%=========================================================================
%===========               FUNCTIONS FOR LISTS                 =========== 
%=========================================================================


% -------------- INITLIST -------------
function T=initlist(T)
%disp('---- INITLIST ----')

th=findobj('Tag',['TT_main_' T.TableName]);

if isempty(th)   
    
    if isunix
        fontname = 'Monospaced';
    else
        fontname = 'FixedWidth';
    end
    
    bgrh = uicontrol('Parent',T.window_h, ... 
        'Style','Listbox', ...
        'BackgroundColor',[1 1 1], ...
        'ForegroundColor',[0 0 0],...
        'FontName',fontname,...
        'FontWeight','normal',...
        'Position',T.Position, ...
        'FontSize',T.FontSize,...
        'CallBack','zz_tabletool(7.1);',...
        'Max',2,...
        'Min',0,...
        'Tag',['TT_main_' T.TableName],...
        'Interruptible','off');
    % Interruptible set to off to ensure callback finishes before new call
    % from same list.
    
    T.tabletype=3;
    
    set(bgrh,'UserData',T)
    T=fillist(T.TableName);     
    set(bgrh,'Value',1); drawnow; % Ensures top of list is shown before setlistselection
    T=setlistselection(T.TableName);
    T=setuimenus(T.TableName);
    
end


% ------------ FILLIST -------------
function T=fillist(tablename)

tableh=findobj('Tag',['TT_main_' tablename]);
T=get(tableh,'UserData');

adjlines=[]; % lines to adjust (values and column headers)
      
% Sets rowtitles
for x=1:T.n_rows rowtitles{x}=['Row ' num2str(x)]; end
if isfield(T,'RowName') 
    if iscell(T.RowName)
        [rowtitles{:}]=deal(''); for x=1:length(T.RowName) rowtitles{x}=T.RowName{x}; end
    end
    if isempty(T.RowName) rowtitles=[]; end
end
if isempty(rowtitles) coff=0; else coff=1; end

% Sets columntitles
for x=1:T.n_cols columntitles{x}=['Col ' num2str(x)]; end
if isfield(T,'ColumnName') 
    if iscell(T.ColumnName)
        [columntitles{:}]=deal(''); for x=1:length(T.ColumnName) columntitles{x}=T.ColumnName{x}; end
    end
    if isempty(T.ColumnName) columntitles=[]; end
end

% ----- creates outstr -----
horseps=[]; roff=0;
% Headers
if isfield(T,'HeaderText')
    for x=1:numel(T.HeaderText)
        outstr{x,1}=T.HeaderText{x}; roff=roff+1; 
    end
    roff=roff+1;
    horseps=[horseps roff];
end

% sets columntitles
if ~isempty(columntitles)
    if length(columntitles)>0 adjlines(numel(adjlines)+1)=roff+1; end
    for cno=1:length(columntitles)
        outstr{roff+1,cno+coff}=columntitles{cno}; 
    end
    roff=roff+2; horseps=[horseps roff];
end

% Sets rowtitles
if ~isempty(rowtitles)
    for rno=1:length(rowtitles)
        outstr{rno+roff,1}=rowtitles{rno}; 
    end
    rowtitoffs=1; % Used as offset for removing column if ShowListSel==0
else
    rowtitoffs=0;
end

adjlines=[adjlines roff+1:T.n_rows+roff];

% Sets Data
for cno=1:T.n_cols
    
    if iscell(T.ColumnFormat{cno})
        rightadjust(cno+coff)=0;
        for rno=1:T.n_rows
            outstr{rno+roff,cno+coff}=T.Data{rno,cno};
        end
    end
    if isnumeric(T.ColumnFormat{cno})
        rightadjust(cno+coff)=1;
        for rno=1:T.n_rows
            outstr{rno+roff,cno+coff}=num2str(T.Data{rno,cno},T.ColumnFormat{cno});
        end
    end
    if ischar(T.ColumnFormat{cno}) && strncmpi(T.ColumnFormat{cno},'numeric',3)
        rightadjust(cno+coff)=1;
        for rno=1:T.n_rows
            outstr{rno+roff,cno+coff}=num2str(T.Data{rno,cno});
        end
    end
    if ischar(T.ColumnFormat{cno}) && strncmpi(T.ColumnFormat{cno},'integer',3)
        rightadjust(cno+coff)=1;
        for rno=1:T.n_rows
            outstr{rno+roff,cno+coff}=num2str(round(T.Data{rno,cno}));
        end
    end
    if ischar(T.ColumnFormat{cno}) && strncmpi(T.ColumnFormat{cno},'%',1)
        rightadjust(cno+coff)=1;
         for rno=1:T.n_rows
             outstr{rno+roff,cno+coff}=num2str(T.Data{rno,cno},T.ColumnFormat{cno});
        end
    end
    if ischar(T.ColumnFormat{cno}) && sum(strncmpi(T.ColumnFormat{cno},{'string','char','text'},3))>0     
        rightadjust(cno+coff)=0;
         for rno=1:T.n_rows
            outstr{rno+roff,cno+coff}=num2str(T.Data{rno,cno});
        end
    end
    if ischar(T.ColumnFormat{cno}) && sum(strncmpi(T.ColumnFormat{cno},'button',3))>0     
        rightadjust(cno+coff)=0;
         for rno=1:T.n_rows
            outstr{rno+roff,cno+coff}=['[' num2str(T.Data{rno,cno}) ']'];
        end
    end    
    if ischar(T.ColumnFormat{cno}) && sum(strncmpi(T.ColumnFormat{cno},{'logical','binary','excl_logical','excl_lbinary'},3))>0 
        rightadjust(cno+coff)=0.5;
        for rno=1:T.n_rows
            if T.Data{rno,cno}
                %disp(['--- row no ' num2str(rno)])
                outstr{rno+roff,cno+coff}='X';
            else
                outstr{rno+roff,cno+coff}=' ';
            end
        end
    end  
    if ischar(T.ColumnFormat{cno}) && strncmpi(T.ColumnFormat{cno},'color',3)   
        T.N.newtabcolformat{cno}='char';
        rightadjust(cno+coff)=1;
        for rno=1:T.n_rows
            colr=T.Data{rno,cno}*255;       
            if isnumeric(T.Data{rno,cno}) && numel(T.Data{rno,cno})==3
                outstr{rno+roff,cno+coff}=[num2str(colr(1)) ':' num2str(colr(2)) ':' num2str(colr(3))];
            else           
                outstr{rno+roff,cno+coff}='-';
            end
        end 
    end    
end

T.L.firstrowindex=roff+1;
T.L.lastrowindex=roff+rno;

% Footers
if isfield(T,'FooterText')
    lineno=roff+rno+1;
    horseps=[horseps lineno];
    for x=1:numel(T.FooterText)
        outstr{lineno+x,1}=T.FooterText{x}; 
    end 
end

% Creates stringarray for output
ind=cellfun('isempty',outstr); outstr(ind)={' '};
[n_outrows,n_outcols]=size(outstr);

% Removes listsel. column
if isfield(T,'ShowListSel') && ~T.ShowListSel && isfield(T,'ListSelCol') && T.ListSelCol<=n_outcols
    outstr(:,T.ListSelCol+rowtitoffs)=[];
    rightadjust(T.ListSelCol+rowtitoffs)=[];
    n_outcols=n_outcols-1;
end

% Create matrix for table
for cno=1:n_outcols    
    switch rightadjust(cno)
        case 1
            omat{cno}=strjust(strvcat(outstr(adjlines,cno)),'right');
        case 0.5
            omat{cno}=strjust(strvcat(outstr(adjlines,cno)),'center');
        case 0
            omat{cno}=strvcat(outstr(adjlines,cno));
    end
end
STRS=[];
vertsepstr=repmat('  ',numel(adjlines),1);
for cno=1:numel(omat)
    STRS=[STRS omat{cno} vertsepstr];
end

% put strings together
[nrows,ncols]=size(outstr);
Tstr=cell(0); maxchar=0;
for rno=1:nrows
    ind=find(rno==adjlines);
    if isempty(ind)
        Tstr(rno,1)=deblank(outstr(rno,1));
        maxchar=max([maxchar numel(Tstr{rno})]);
    else
        Tstr{rno,1}=deblank(STRS(ind,:));
        maxchar=max([maxchar numel(Tstr{rno})]);
    end
end

% Add separators
seps=find(cellfun('isempty',Tstr));
sepstr=repmat('-',1,maxchar);
for x=1:numel(seps)
    Tstr{seps(x)}=sepstr;
end

set(tableh,'String',Tstr,'Userdata',T)


% ------------- SETLISTSELECTION --------------
function T=setlistselection(tablename)
%disp('--- setlistselection ---')

tableh=findobj('Tag',['TT_main_' tablename]);
T=get(tableh,'UserData');

selectedcells=[];
if isfield(T,'ListSelCol') && isnumeric(T.ListSelCol) && ~isempty(T.ListSelCol)
    datatype=T.datatype{T.ListSelCol};
    if strncmpi(datatype,'logical',3) || strncmpi(datatype,'binary',3) 
        set(tableh,'Max',2,'Min',0)
        for rno=1:T.n_rows
            if T.Data{rno,T.ListSelCol}>0
                selectedcells=[selectedcells rno];
            end
        end
        selectedcells=selectedcells+T.L.firstrowindex-1;
        set(tableh,'Value',selectedcells);
    end
    if strncmpi(datatype,'excl_logical',3) || strncmpi(datatype,'excl_binary',3) 
        set(tableh,'Max',1,'Min',0)
        for rno=1:T.n_rows
            if T.Data{rno,T.ListSelCol}>0
                selectedcells=[selectedcells rno];
            end
        end
        selectedcells=selectedcells+T.L.firstrowindex-1;
        % Sets selection to line 1 for single selection listbox if noting is selected
        if isempty(selectedcells) selectedcells=1; end
        selectedcells=selectedcells(1); % Ensures single value is set
    end    
end

set(tableh,'Value',selectedcells)


% ------------- LISTSELECTION --------------
function T=listselection(T)
%disp('---- listselection ----');

tableh=findobj('Tag',['TT_main_' T.TableName]);
T=get(tableh,'UserData');

if isfield(T,'ListSelCol') && isnumeric(T.ListSelCol) && ~isempty(T.ListSelCol)
    
    previousset=cell2mat(T.Data(:,T.ListSelCol));

    positions=get(tableh,'Value');
    validpositions=find(positions>=T.L.firstrowindex & positions<=T.L.lastrowindex);

    values=positions(validpositions)-T.L.firstrowindex+1;
    datatype=T.datatype{T.ListSelCol};
    if isempty(values) && (strncmpi(datatype,'excl_logical',3) || strncmpi(datatype,'excl_binary',3))
        if abs(T.L.firstrowindex-positions(1)) < abs(T.L.lastrowindex-positions(1))
            values=true;
        else
            values=T.n_rows;
        end 
    end

    currentset=false(T.n_rows,1); 
    currentset(values)=true;
    changedrow=(find(abs(currentset-previousset)));
    
    if ~isempty(changedrow)    
        [T.Data{:,T.ListSelCol}]=deal(false); [T.Data{values,T.ListSelCol}]=deal(true);
        T.lastedit_value=currentset(changedrow);
        T.lastedit_previousvalue=previousset(changedrow);
        T.lastedit_colindex=repmat(T.ListSelCol,1,length(changedrow));
        T.lastedit_rowindex=changedrow;
        T.lastedit_datatype=T.datatype{T.ListSelCol};
        T.userinput(changedrow,T.CellEditCallback)=1;
        T.changedtime(changedrow,T.CellEditCallback)=now;
        set(tableh,'UserData',T)
        
        eval(T.CellEditCallback); % Runs callback  
        try
            uicontrol(gco); % Returns control to list, not supportd in Matlab 6.5
        catch
            
        end
    end
    
    % Updates if hidden list selection and illegal lines (handled elsewhere
    % if visible listselection)
    if isfield(T,'ShowListSel') && ~T.ShowListSel && numel(validpositions) <numel(positions)
        setlistselection(T.TableName);
    end   
end


% ------------- CLEARALL --------------
function clearall
%disp('--- clearall ---')

ah=get(gcf,'Children');
for x=1:length(ah)
    %tags{y}=get(ah(y),'Tag');
    if strncmp('TT_',get(ah(x),'Tag'),3)
        delete(ah(x))
    end
end


% -------------- DRAWWIN -------------
function h0=drawwin(winpar)

% Defines userdata
Userdata.callerwin_h=gcbf;
Userdata.callerwin_tag=get(gcbf,'Tag');
Userdata.callerwin_updfcn=get(gcbf,'ResizeFcn');
Userdata.callerwin_pos=get(gcbf,'Position');
Userdata.callerwin_units=get(gcbf,'Units');
Userdata.callerwin_color=get(gcbf,'Color');
Userdata.callerwin_currentaxes=get(gcbf,'CurrentAxes');
Userdata.callerwin_currentaxes_tag=get(Userdata.callerwin_currentaxes,'Tag');
Userdata.callerobj_h=gcbo;
Userdata.callerobj_tag=get(gcbo,'Tag');
calleruserdat=get(gcbf,'UserData');
Userdata.pixwidth=calleruserdat.pixwidth;
Userdata.pixheight=calleruserdat.pixheight;
Userdata.pointwidth=calleruserdat.pointwidth;
Userdata.pointheight=calleruserdat.pointheight;
  
h0 = figure('Color',get(gcbf,'Color'), ...
    'PaperPosition',[18 180 576 432], ...
    'PaperUnits','points', ...
    'Units','normalized',...
    'Position',winpar{3}, ...
	'Renderer','OpenGL', ...
	'RendererMode','auto', ...
	'Tag',winpar{1}, ...
    'Name',winpar{2},...
    'Resize','off',...
    'NumberTitle','off',...
    'Handlevisibility','on',...
    'Menubar','none',...
    'UserData',Userdata,...
    'ToolBar','none');

delstr=['if ~isempty(findobj(''Tag'',''' Userdata.callerwin_tag ''')) ' 'figure(findobj(''Tag'',''' Userdata.callerwin_tag ''')); end'];
set(h0,'DeleteFcn',delstr)




